/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

#include <assert.h>
#include <memory.h>
#include <fcntl.h>

#include <typedefs.h>

#include "pci.h"
#include "b_pci.h"
#include "b_cmd.h"
#include "b_hif.h"
#include "b_io.h"
#include "b_ioctl.h"

extern b_handlestype handle_array[];

static b_errtype BestPCIDWGet(
    b_int32 Reg_N,              /* offset into config. space */
    b_portnumtype OsHandle,
    b_int32 * pResult);
static b_errtype BestPCIDWSet(
    b_int32 Reg_N,
    b_portnumtype OsHandle,
    b_int32 value);

extern b_int32 GlobalBaseAddr;

/* --------------------------------------------------------------------------
 * Parallel port
 * -------------------------------------------------------------------------- */

long int BestOpenParallel(int num)
{
  return 42L;
}

void BestCloseParallel(b_portnumtype portnumber)
{
  return;
}

b_errtype BestParDeviceConnect(b_portnumtype OsHandle)
{
  return B_E_OK;
}

b_errtype BestParCheckConnection(b_portnumtype OsHandle)
{
  return B_E_OK;
}

void BestParReleaseConnection(b_portnumtype portnumber)
{
}

b_errtype BestParPortTimeoutSet(b_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  return B_E_OK;
}

/* --------------------------------------------------------------------------
 * Read/write to card
 * -------------------------------------------------------------------------- */

b_errtype BestBasicRead(
    b_handletype BestHandleArrIndx,
    b_int8 * pData,       /* caller's buffer */
    b_int32 NumBytes,     /* number of bytes to read */
    b_int8 DataEltWidth)  /* data element width in bytes */

{
  b_errtype	err = B_E_OK;
  b_int32 bytes_read, bytes_left;

  assert(pData && NumBytes < 0x0000ffff);
  
  switch (handle_array[BestHandleArrIndx].port) 
  {
   case B_PORT_OFFLINE:
     memset( pData, 0, NumBytes );
     break;

   case B_PORT_PCI_CONF:
   case B_PORT_PCI_IO:
   case B_PORT_RS232:
     bytes_left=NumBytes;
     while (bytes_left) 
     {
       bytes_read=read(handle_array[BestHandleArrIndx].portnumber,pData,bytes_left);
       bytes_left-=bytes_read;
       pData+=bytes_read;
     }
     return B_E_OK;
     break;
  
   case B_PORT_PARALLEL:
   case B_PORT_FASTHIF:
    err = B_E_BAD_HANDLE;
    break;
  }
  
  return err;
}


b_errtype BestBasicWrite(
    b_handletype BestHandleArrIndx,
    b_int8 * pData,       /* caller's buffer */
    b_int32 NumBytes,     /* number of bytes to write */
    b_int8 DataEltWidth)  /* data element width in bytes */

{
  b_errtype err;
  b_int32 bytes_left,bytes_written;

  assert(pData && NumBytes < 0x0000ffff);

  switch (handle_array[BestHandleArrIndx].port) {
   case B_PORT_OFFLINE:
     err = B_E_OK;
     break;

   case B_PORT_PCI_CONF:
   case B_PORT_PCI_IO:
   case B_PORT_RS232:
     bytes_left=NumBytes;
     while (bytes_left) {
       bytes_written=write(handle_array[BestHandleArrIndx].portnumber,pData,bytes_left);
       bytes_left-=bytes_written;
       pData+=bytes_written;
     }
     return B_E_OK;
     break;
  
   case B_PORT_PARALLEL:
   case B_PORT_FASTHIF:
    err = B_E_BAD_HANDLE;
    break;
  }
      
  return err;
}

/* --------------------------------------------------------------------------
 * PCI port
 * -------------------------------------------------------------------------- */

b_errtype BestPCISetRegwidth(b_portnumtype OsHandle, int regwidth)
{
  ioctl(OsHandle,IOCTL_NTIO_SET_REGWIDTH_PCI,regwidth);

  return B_E_OK;
}

b_errtype BestPCIDeviceConnect(b_portnumtype OsHandle)
{
  if (BestPCIDWSet(PCI_CONNECT_CMD, OsHandle, PCI_CONNECT_CMD_BIT) != B_E_OK)
  {
    /* TODO: error param */
    return B_E_ERROR;
  }
 
  return B_E_OK;
}

b_errtype BestPCICheckConnection(b_portnumtype OsHandle)
{
  b_errtype lErrCode;
  b_int32 Result;

  if (B_E_OK != (lErrCode = BestPCIDWGet(PCI_CONNECT_STATUS, OsHandle, &Result)))
    return lErrCode;
 
  if ((Result & PCI_CONNECT_STATUS_BIT) == PCI_CONNECTION_CLOSED)
    return B_E_NOT_CONNECTED;
 
  return B_E_OK;
}

void BestPCIReleaseConnection(b_portnumtype portnumber)
{
  (void) BestPCIDWSet(PCI_CONNECT_CMD, portnumber, 0x0);
  return;
}

b_errtype BestPciPortTimeoutSet(b_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  ioctl(OsHandle,IOCTL_NTIO_SET_TIMEOUT,pCallersTimeouts);
  return B_E_OK;
}

b_errtype BestPciDevMultiFuncCheck(b_int32 dwHPSlotId, b_bool * fIsMultiFunc)
{
  b_int32 Result;

  *fIsMultiFunc = 0;
 
  if ( B_E_OK != BestGenericPCIDWGet(dwHPSlotId, PCI_MF_OFFSET, &Result))
    return B_E_NO_BEST_PCI_DEVICE_FOUND;
 
  *fIsMultiFunc = ((Result & PCI_MF_BIT) != 0);
  return B_E_OK;
}

b_errtype BestPCIMailboxWrite(b_int32 dwHPSlotId, b_int32 value)
{
  b_int32 statusdata;
  BEST_TIMER tmr;
  b_errtype err;

  /* init timeout */
  BestStartTimeout(TIMEOUT_MAILBOX_PCI_WRITE, &tmr);
 
  do
  {
    /* get Mailbox status */
    if (B_E_OK != (err = BestGenericPCIDWGet(dwHPSlotId, PCI_MBOX_STATUS, &statusdata)))
      return err;
 
    if ( BestIsTimeoutDone(&tmr) ) {
      /* TODO: error param */
      return B_E_ERROR;  /* a timeout here is an error */
    }
 
  } while ((statusdata & PCI_MBOX_SEND_FULL_BIT) != 0);
 
  /* write data to Mailbox */
  return BestGenericPCIDWSet(dwHPSlotId, PCI_MBOX_DATA, value);
}

b_errtype BestPCIMailboxRead(b_int32 dwHPSlotId, b_int32 * pResult)
{
  b_int32 statusdata;
  BEST_TIMER  tmr;
  b_errtype err;
  assert(pResult && "Null passed to BestPCIMailboxRead()");

  /* init timeout */
  BestStartTimeout(TIMEOUT_MAILBOX_PCI_READ, &tmr);
 
  do
  {
    /* get status from Config */
    if (B_E_OK != (err = BestGenericPCIDWGet(dwHPSlotId, PCI_MBOX_STATUS, &statusdata)))
      return err;
 
    if ( BestIsTimeoutDone(&tmr) ) {
      /* TODO: error param */
      return B_E_ERROR;               /* a timeout here is an error */
    }
 
  } while ((statusdata & PCI_MBOX_DATA_VALID_BIT) == 0);
 
  /* get data from data register */
  if (B_E_OK != (err = BestGenericPCIDWGet(dwHPSlotId, PCI_MBOX_DATA, &statusdata)))
    return err;
 
  /* get data from data register */
  *pResult = statusdata;
  
  /* clear the data valid bit */
  return BestGenericPCIDWSet(dwHPSlotId, PCI_MBOX_STATUS, PCI_MBOX_DATA_VALID_BIT);
}

b_errtype BestOpenPCI(int portnum, b_portnumtype *pOsHandle)
{
  bestpci_general_info geninf;
  bestpci_device_info devinf;
  int fd,devnr;
  FILE *fd1;
  char path[1024];
  char *temp=(char *)malloc(1024);
  char **ptemp=&temp;
  char name[1024];
  
  strcpy(path,"/dev");

  // First search for the location of the devfs
  fd1=fopen("/etc/mtab","r");
  while (!feof(fd1)) 
  {
    fgets(temp,1024,fd1);
    if (strstr(temp," devfs ")) 
    {
      strsep(ptemp," ");
      strcpy(path,strsep(ptemp," "));
    }
  }
  fclose(fd1);
  free(temp);

  sprintf(name,"%s/best/pci0",path);

  fd=open(name,O_RDONLY);
  if (fd<0) 
  {
    *pOsHandle=INVALID_OS_HANDLE;
    return B_E_ERROR;
  }

  ioctl(fd,0,&geninf);
  
  devnr=0;
  while ((devnr<geninf.dev_count) && (devinf.slot_id.u.AsULONG != portnum))
  {
    ioctl(fd,devnr,&devinf);
    devnr++;
  }
  devnr--;
  close(fd);
  
  if (devinf.slot_id.u.AsULONG == portnum) 
  {
    sprintf(name,"%s/best/pci%d",path,devnr);
    *pOsHandle=(b_portnumtype)open(name,O_RDWR);
    if (&pOsHandle>=0)
    {
      return B_E_OK;
    }
  }
    
  *pOsHandle=INVALID_OS_HANDLE;

  return B_E_ERROR;
}

b_errtype BestClosePCI(b_portnumtype OsHandle)
{
  close(OsHandle);

  return B_E_OK;
}


b_errtype BestGenericPCIDWSet(b_int32 DeviceId,
    b_int32 Reg_N,
    b_int32 value)
{
  /*
   DeviceID consists of BusNo(8bit),Slotnumber(5 bits) and functionnumber (3 bit):
   devid=busnumber<<8 | slotnumber<<3 | function
  */
  
  int fd,ret;

  if (BestOpenPCI(DeviceId,&fd) == B_E_OK)
  {
    ret=BestPCIDWSet(Reg_N,fd,value);
    BestClosePCI(fd);
    return ret;
  }

  return B_E_NO_BEST_PCI_DEVICE_FOUND;
}

b_errtype BestGenericPCIDWGet(b_int32 DeviceId,
    b_int32 Reg_N,
    b_int32 * value)
{
  /*
   DeviceID consists of BusNo(8bit),Slotnumber(5 bits) and functionnumber (3 bit):
   devid=busnumber<<8 | slotnumber<<3 | function
  */

  int a,fd;

  if (BestOpenPCI(DeviceId,&fd) == B_E_OK) 
  {
    if (BestPCIDWGet(Reg_N,fd,value) == B_E_OK )
    {
      BestClosePCI(fd);
      return B_E_OK;
    }
  }
  return B_E_NO_BEST_PCI_DEVICE_FOUND;
}

b_errtype BestOpenIO(int portnum, b_portnumtype *pOsHandle)
{
  return B_E_ERROR;
}

b_errtype BestCloseIO(b_portnumtype portnumber)
{
  return B_E_ERROR;
}

static b_errtype BestPCIDWGet(
    b_int32 Reg_N,              /* offset into config. space */
    b_portnumtype OsHandle,
    b_int32 * pResult)
{
  b_int32 data=Reg_N;

  ioctl(OsHandle, IOCTL_NTIO_GET_CONFIG_DW, &data);
  *pResult=data;

  return B_E_OK;
}

static b_errtype BestPCIDWSet(
    b_int32 Reg_N,
    b_portnumtype OsHandle,
    b_int32 value)
{
  b_configdrivertype WriteBuffer;

  WriteBuffer.Offset = Reg_N;
  WriteBuffer.Data.Data = value;
  ioctl(OsHandle,IOCTL_NTIO_SET_CONFIG_DW, &WriteBuffer);
  return B_E_OK;
}

/* --------------------------------------------------------------------------
 * Fast HIF dummy functions
 * -------------------------------------------------------------------------- */

b_errtype BestOpenHIF(b_int32 dwCardId,
    b_int32 * pHPSlotId,
    b_portnumtype * pOsHandle)
{
  return B_E_ERROR;
}

b_errtype BestCloseHIF(b_portnumtype OsHandle)
{
  return B_E_ERROR;
}

b_errtype BestHIFSetRegwidth(b_portnumtype OsHandle, int iRegWidth)
{
  return B_E_ERROR;
}

b_errtype BestHIFDeviceConnect(b_portnumtype OsHandle)
{
  return B_E_ERROR;
}

b_errtype BestHIFCheckConnection(b_portnumtype OsHandle)
{
  return B_E_ERROR;
}

b_errtype BestHIFIsDisconnected(b_portnumtype OsHandle)
{
  return B_E_ERROR;
}

void BestHIFReleaseConnection(b_portnumtype OsHandle)
{
}

b_errtype BestHifPortTimeoutSet(b_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  return B_E_ERROR;
}
